<?php
session_start();
require_once "db.php"; 
date_default_timezone_set("Africa/Nairobi");

// Ensure user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$user_id = $_SESSION['user_id'];

// 1. Get my referral code
$stmt = $conn->prepare("SELECT referral_code FROM users WHERE id=?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($referral_code);
$stmt->fetch();
$stmt->close();

// 2. Count direct referrals
$stmt = $conn->prepare("SELECT COUNT(*) FROM users WHERE referred_by=?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($direct_referrals);
$stmt->fetch();
$stmt->close();

// 3. Total earnings from referral_bonuses
$stmt = $conn->prepare("SELECT IFNULL(SUM(amount),0) FROM referral_bonuses WHERE to_user_id=?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($total_earnings);
$stmt->fetch();
$stmt->close();

// 4. Team members with earnings
$sql = "
    SELECT u.id, u.username, u.created_at, 
           IFNULL(SUM(rb.amount),0) AS earnings
    FROM users u
    LEFT JOIN referral_bonuses rb 
        ON rb.from_user_id = u.id AND rb.to_user_id = ?
    WHERE u.referred_by = ?
    GROUP BY u.id, u.username, u.created_at
    ORDER BY u.created_at DESC
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ii", $user_id, $user_id);
$stmt->execute();
$result = $stmt->get_result();
$team_members = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>FCB - My Team</title>
  <style>
    @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800&display=swap');
    :root{
      --bg1:#1ca68f;
      --bg2:#2aa78d;
      --accent:#1fbf9a;
      --accent-dark:#16826f;
    }
    *{margin:0;padding:0;box-sizing:border-box}
    body{
      font-family:Inter,system-ui;
      background:linear-gradient(180deg,var(--bg1),var(--bg2));
      display:flex;justify-content:center;
      padding:24px 12px;
      color:#fff;
    }
    .phone{
      width:390px;min-height:780px;
      background:rgba(255,255,255,0.02);
      border-radius:28px;
      padding:18px 16px 98px;
      box-shadow:0 18px 40px rgba(5,40,35,0.45);
      position:relative;
    }
    h1{text-align:center;font-size:22px;margin-bottom:20px;font-weight:700}
    
    .invite-card{
      background:rgba(0,0,0,0.3);
      padding:16px;
      border-radius:16px;
      margin-bottom:18px;
      text-align:center;
    }
    .invite-card h2{font-size:18px;margin-bottom:10px}
    .invite-code{
      font-size:20px;
      font-weight:800;
      color:var(--accent);
      letter-spacing:2px;
      margin-bottom:12px;
    }
    .btn{
      display:inline-block;
      padding:10px 14px;
      border-radius:12px;
      border:none;
      cursor:pointer;
      font-size:14px;
      font-weight:600;
      margin:6px 4px;
    }
    .btn-primary{background:linear-gradient(180deg,var(--accent),var(--accent-dark));color:#fff}
    .btn-outline{background:transparent;border:1.5px solid var(--accent);color:var(--accent)}

    .qr-box{
      background:#fff;padding:12px;border-radius:12px;
      display:flex;justify-content:center;
      margin-top:14px;
    }
    .stats{
      display:grid;
      grid-template-columns:1fr 1fr;
      gap:14px;
      margin:20px 0;
    }
    .stat{
      background:rgba(0,0,0,0.25);
      padding:14px;
      border-radius:12px;
      text-align:center;
    }
    .stat .num{font-size:20px;font-weight:700;color:var(--accent)}
    .team-list{
      margin-top:10px;
    }
    .team-list table{
      width:100%;
      border-collapse:collapse;
      font-size:14px;
    }
    .team-list th,.team-list td{
      padding:8px;
      border-bottom:1px solid rgba(255,255,255,0.1);
      text-align:left;
    }
    .team-list th{color:var(--accent);font-weight:600}

    /* bottom nav */
    .bottom-nav{
      position:fixed;left:50%;transform:translateX(-50%);
      bottom:12px;width:calc(100% - 36px);max-width:366px;
      background:linear-gradient(180deg,rgba(0,0,0,0.02),rgba(0,0,0,0.03));
      backdrop-filter:blur(6px);
      border-radius:14px;padding:10px 12px;
      display:flex;align-items:center;justify-content:space-around;
    }
    .nav-item{flex:1;display:flex;flex-direction:column;align-items:center;padding:8px 6px;border-radius:10px;color:#dffaf4;font-size:13px;cursor:pointer}
    .nav-item .icon{width:44px;height:44px;border-radius:10px;display:flex;align-items:center;justify-content:center;background:rgba(255,255,255,0.02)}
    .nav-item.active{background:linear-gradient(180deg,rgba(0,0,0,0.07),rgba(0,0,0,0.03))}
    .nav-item svg{width:22px;height:22px;fill:#dffaf4}
    .nav-item.active svg{fill:var(--accent)}
  </style>
</head>
<body>
  <div class="phone">
    <h1>Smart360</h1>

    <!-- Invitation -->
    <div class="invite-card">
      <h2>Invitation Code</h2>
      <div class="invite-code" id="inviteCode"><?= htmlspecialchars($referral_code) ?></div>
      <button class="btn btn-primary" onclick="copyCode()">Copy Code</button>
      <button class="btn btn-outline" onclick="copyLink()">Copy Link</button>

      <div class="qr-box">
        <canvas id="qrcode"></canvas>
      </div>
    </div>

    <!-- Stats -->
    <div class="stats">
      <div class="stat">
        <div class="num"><?= $direct_referrals ?></div>
        <div>Direct Referrals</div>
      </div>
      <div class="stat">
        <div class="num"><?= number_format($total_earnings,2) ?></div>
        <div>Total Earnings (USDT)</div>
      </div>
    </div>

    <!-- Team List -->
    <div class="team-list">
      <h2 style="font-size:16px;margin-bottom:8px">Team Members</h2>
      <table>
        <thead>
          <tr>
            <th>User</th>
            <th>Joined</th>
            <th>Earnings</th>
          </tr>
        </thead>
        <tbody>
          <?php if (count($team_members) > 0): ?>
            <?php foreach ($team_members as $tm): ?>
              <tr>
                <td><?= htmlspecialchars($tm['username']) ?></td>
                <td><?= htmlspecialchars(date("Y-m-d", strtotime($tm['created_at']))) ?></td>
                <td><?= number_format($tm['earnings'],2) ?></td>
              </tr>
            <?php endforeach; ?>
          <?php else: ?>
            <tr><td colspan="3" style="text-align:center;color:#ccc">No team members yet</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

  <!-- Bottom Navigation -->
  <div class="bottom-nav">
    <div class="nav-item" onclick="location.href='home.php'">
      <div class="icon"><svg viewBox="0 0 24 24"><path d="M3 10L12 3l9 7v11a1 1 0 0 1-1 1h-6v-7h-4v7H4a1 1 0 0 1-1-1V10z"/></svg></div>
      <div style="margin-top:6px">Home</div>
    </div>
    <div class="nav-item" onclick="location.href='earn.php'">
      <div class="icon"><div style="width:24px;height:24px;background:linear-gradient(180deg,#1fbf9a,#16826f);border-radius:50%;display:flex;align-items:center;justify-content:center;font-weight:700;color:#fff">T</div></div>
      <div style="margin-top:6px">Earn</div>
    </div>
    <div class="nav-item" onclick="location.href='invest.php'">
      <div class="icon"><svg viewBox="0 0 24 24"><path d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7z"/></svg></div>
      <div style="margin-top:6px">Invest</div>
    </div>
    <div class="nav-item active" onclick="location.href='team.php'">
      <div class="icon"><svg viewBox="0 0 24 24"><path d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5s-3 1.34-3 3 1.34 3 3 3zM8 11c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5C15 14.17 10.33 13 8 13zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z"/></svg></div>
      <div style="margin-top:6px">Team</div>
    </div>
    <div class="nav-item" onclick="location.href='profile.php'">
      <div class="icon"><svg viewBox="0 0 24 24"><path d="M12 12c2.7 0 5-2.3 5-5s-2.3-5-5-5-5 2.3-5 5 2.3 5 5 5zm0 2c-3.3 0-10 1.7-10 5v3h20v-3c0-3.3-6.7-5-10-5z"/></svg></div>
      <div style="margin-top:6px">Profile</div>
    </div>
  </div>

  <!-- Scripts -->
  <script src="https://cdn.jsdelivr.net/npm/qrcode/build/qrcode.min.js"></script>
  <script>
    const inviteCode = document.getElementById("inviteCode").innerText;
    const inviteLink = "https://fcb.com/register?code=" + inviteCode;

    function copyCode(){
      navigator.clipboard.writeText(inviteCode);
      alert("Invitation Code copied: " + inviteCode);
    }
    function copyLink(){
      navigator.clipboard.writeText(inviteLink);
      alert("Invitation Link copied: " + inviteLink);
    }

    // Generate QR code
    QRCode.toCanvas(document.getElementById('qrcode'), inviteLink, { width:150 }, function (error) {
      if (error) console.error(error)
    })
  </script>
</body>
</html>
